{*******************************************************************************
  Print Logic:

  The first thing you need to do is to write your printing logic. You do this by
  handling events on a PrintDocument. When the PrintDocument.Print() method is
  called, the following events are raised:

  1. BeginPrint
  2. PrintPage (one or more)
  3. EndPrint

  The PrintPage event arguments type (PrintPageEventArgs) has a HasMorePages property.
  If this is set to true when your event handler returns, PrintDocument sets up a
  new page and raises the PrintPage event again.

  Therefore, the logic in your PrintPage event handler is basically the following:

  1. Print the contents of the page using the information in the event arguments.
  The event arguments contain the Graphics for the printer, the PageSettings for
  that page, the bounds of the page, and the size of the margins.

  2. Determine whether there are more pages to print.
  3. If there are more pages, set HasMorePages to true.
  4. If there are not more pages, set HasMorePages to false.

  In the simplest case, you can create a PrintDocument and handle the
  PrintPage event as part of the Form that contains the print request.

********************************************************************************}

program Print;

{%DelphiDotNetAssemblyCompiler '$(SystemRoot)\microsoft.net\framework\v1.1.4322\System.Drawing.dll'}
{%DelphiDotNetAssemblyCompiler '$(SystemRoot)\microsoft.net\framework\v1.1.4322\System.Windows.Forms.dll'}

uses
  System.IO,
  System.ComponentModel,
  System.Windows.Forms,
  System.Drawing,
  System.Drawing.Printing;

type
  TForm        = System.Windows.Forms.Form;
  TButton      = System.Windows.Forms.Button;

  TSize          = System.Drawing.Size;
  TPoint         = System.Drawing.Point;
  TFont          = System.Drawing.Font;
  TStringFormat  = System.Drawing.StringFormat;
  TPrintDocument = System.Drawing.Printing.PrintDocument;

  TStreamReader  = System.IO.StreamReader;

type
  { Application main form }
  TMainForm = class(TForm)
  private
    FPrintFont: TFont;
    FStreamReader: TStreamReader;
    Button1: TButton;
    procedure InitComponents;
    procedure Button1Click(Sender: TObject; e: EventArgs);
    procedure PrintPage(Sender: TObject; pe: PrintPageEventArgs);
  public
    constructor Create(Owner: TObject);
  end;

{ TMainForm }

constructor TMainForm.Create(Owner: TObject);
begin
  inherited Create;
  InitComponents;
end;

procedure TMainForm.InitComponents;
begin
  SuspendLayout;

  Text:= 'Print Logic';
  AutoScaleBaseSize:= TSize.Create(5, 13);
  ClientSize:= TSize.Create(500, 380);

  Button1:= TButton.Create;
  Button1.Top:= 5;
  Button1.Left:= 5;
  Button1.Width:= 100;
  Button1.ImageAlign:= ContentAlignment.MiddleLeft;
  Button1.FlatStyle:= FlatStyle.Flat;
  Button1.TabIndex:= 0;
  Button1.Text:='Print';
  Button1.Add_Click(Button1Click);
  Controls.Add(Button1);

  ResumeLayout(False);
end;

procedure TMainForm.Button1Click(Sender: TObject; e: EventArgs);
var
  pd: TPrintDocument;
begin
  try
    FStreamReader:= TStreamReader.Create('PrintMe.txt');
    try
      FPrintFont:= TFont.Create('Arial', 10);
      pd:= TPrintDocument.Create; //Assumes the default printer
      pd.add_PrintPage(PrintPage);
      pd.Print;
      MessageBox.Show('Document printed!');
    finally
      FStreamReader.Close;
    end;
  except
    on E: System.Exception do
      MessageBox.Show('An error occurred printing the file - ' + E.Message);
  end;
end;

procedure TMainForm.PrintPage(Sender: TObject; pe: PrintPageEventArgs);
var
  lpp, yPos: Single;
  Count: Integer;
  leftMargin, topMargin: Single;
  line: string;
begin
  count:= 0;
  leftMargin:= pe.MarginBounds.Left;
  topMargin:= pe.MarginBounds.Top;
  line:= '';

  //Work out the number of lines per page
  //Use the MarginBounds on the event to do this
  lpp:= pe.MarginBounds.Height / FPrintFont.GetHeight(pe.Graphics) ;

  //Now iterate over the file printing out each line
  //NOTE WELL: This assumes that a single line is not wider than the page width
  //Check count first so that we don't read line that we won't print
  line:= FStreamReader.ReadLine;
  while (count < lpp) and ( line <>  '') do
  begin
    yPos:= topMargin + (count * FPrintFont.GetHeight(pe.Graphics));
    pe.Graphics.DrawString(line, FPrintFont, Brushes.Black, leftMargin, yPos, TStringFormat.Create);
    Inc(count);
    line:= FStreamReader.ReadLine;
  end;

  //If we have more lines then print another page
  if (line <> '') then
    pe.HasMorePages:= True
  else
    pe.HasMorePages:= False;
end;

{$R *.res}

begin
  Application.Run(TMainForm.Create(nil));
end.
